/* --------------------------------- loop.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* This is the main 'busy-loop' of the simulation.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "fly.h"
#include "hud.h"

static int	FAR world_dynamics (void);
static void	FAR render_bg (OBJECT *pov, int frame);
static void	FAR render_fg (OBJECT *pov, int type);
static void	FAR render_picture (void);
static void	FAR show_picture (void);

extern void FAR
idle_loop (void)
{
	Ulong	t;

	if ((st.flags & SF_SIMULATING) || !(st.flags & SF_INITED))
		return;
	st.flags |= SF_SIMULATING;

	t = Tm->Milli () - st.big_bang - st.present;
	if (3 == st.stereo) {		/* limit refresh rate */
		if (t < REFRESH/2)
			goto ret;
	} else if ((st.flags & SF_DBUFFERING) && !(st.network & NET_ON)) {
		if (t < 10)
			goto ret;
	} else if (t < REFRESH)
		goto ret;

	st.present += t;
	st.ObjectTimeout = st.present + TO_OBJECT;
	st.PlayerTimeout = st.present + TO_PLAYER;
	if (t > 1000L)			/* ignore long pauses */
		st.interval = REFRESH;
	else
		st.interval = (int)t;
	st.dither = rand () % 1000;

	if (!(st.flags & SF_PAUSED)) {
		if (world_dynamics ())
			die ();
		if (objects_dynamics (st.interval))
			die ();
		if (land_update (CV))
			die ();

		render_picture ();	/* build picture */
	}

	show_picture ();	/* refresh screen */

	st.elapsed = (int)(Tm->Milli () - st.big_bang - st.present);

	++st.stats[31];
	st.stats[32] += st.elapsed;
	st.stats[33] += st.video_time;
	st.stats[34] += st.stats[27];
	st.stats[35] += st.stats[28];
	st.stats[36] += st.stats[40];
	st.stats[37] += st.stats[29];

	st.misc[0] = st.elapsed;
	st.misc[1] = st.video_time;
	st.misc[2] = (int)st.stats[27];
	st.misc[3] = (int)st.stats[28];
	st.misc[4] = (int)st.stats[29];
	st.misc[5] = (int)st.stats[40];

	st.stats[27] = 0L;
	st.stats[28] = 0L;
	st.stats[29] = 0L;
	st.stats[40] = 0L;
ret:
	st.flags &= ~SF_SIMULATING;
}

static void FAR
show_buf (BUFFER *b)
{
	register BUFLINE	*p;
	register int	op;
	BUFLINE		*end;
	void		(FAR *MoveTo) (int x1, int y1);
	void		(FAR *DrawTo) (int x2, int y2, Uint c);

	MoveTo = Gr->MoveTo;
	DrawTo = Gr->DrawTo;

	p = b->first;
	end = b->p;
	for (; p < end; ++p) {
		if ((op = p->t) >= 0)
			DrawTo (p->x, p->y, op);
		else if (op == T_MOVE)
			MoveTo (p->x, p->y);
		else if (op == T_MOR || op == T_MXOR || op == T_MSET)
			Gr->WriteMode (op);
		else if (op == T_ELLIPSE) {
			Gr->Ellipse (p->x, p->y, (p+1)->x, (p+1)->y, (p+1)->t);
			++p;
		} else if (op == T_NOP)
			continue;
		else
			break;		/* should not happen! */
	}
}

static void FAR
show_buffer (BUFFER *b)
{
	Gr->WriteMode (T_MSET);			/* initial mode */

	for (; b; b = b->next) {
		show_buf (b);
		sys_poll ();
	}
}

static void FAR
show_buffers (int which)
{
	int	i;

	show_buffer (st.bufs[which]);

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDD_ON))
			continue;
		show_buffer (st.hdd[i].bufs[which]);
	}
}

static void FAR
erase_buf (BUFFER *b)
{
	register BUFLINE	*p;
	register int	op;
	BUFLINE		*end;
	Uint		bg;
	void		(FAR *MoveTo) (int x1, int y1);
	void		(FAR *DrawTo) (int x2, int y2, Uint c);

	MoveTo = Gr->MoveTo;
	DrawTo = Gr->DrawTo;

	if (Gr->Clear) {
		Gr->Clear (CS);
		return;
	}

	bg = CS->BgColor;

	Gr->WriteMode (T_MSET);			/* initial mode */
	p = b->first;
	end = b->p;
	for (; p < end; ++p) {
		if ((op = p->t) >= 0)
			DrawTo (p->x, p->y, bg);
		else if (op == T_MOVE)
			MoveTo (p->x, p->y);
		else if (op == T_MOR || op == T_MXOR || op == T_MSET)
			;
		else if (op == T_ELLIPSE) {
			Gr->Ellipse (p->x, p->y, (p+1)->x, (p+1)->y, bg);
			++p;
		} else if (op == T_NOP)
			continue;
		else
			break;		/* should not happen! */
	}
}

static void FAR
erase_buffer (BUFFER *b)
{
	for (; b; b = b->next) {
		erase_buf (b);
		sys_poll ();
	}
}

static void FAR
erase_buffers (int which)
{
	int	i;

	erase_buffer (st.bufs[which]);

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDD_ON))
			continue;
		erase_buffer (st.hdd[i].bufs[which]);
	}
}

static void FAR
free_buffer (BUFFER *b)
{
	BUFFER	*next;

	for (; b; b = next) {
		next = b->next;
		memory_free (b,
			sizeof (*b) + (b->size - 1) * sizeof (BUFLINE));
		--st.nbuffers;
	}
}

static void FAR
free_buffers (int which)
{
	int	i;

	free_buffer (st.bufs[which]);
	st.bufs[which] = 0;

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDD_ON))
			continue;
		free_buffer (st.hdd[i].bufs[which]);
		st.hdd[i].bufs[which] = 0;
	}
}

extern void FAR
buffers_free (void)
{
	int	i;

	for (i = 0; i < NBUFS; ++i)
		free_buffers (i);

	free_buffer (st.buf[HEAD]);
	st.buf[HEAD] = st.buf[TAIL] = 0;
}

static long FAR
buffer_size (BUFFER *b)
{
	long	l;

	for (l = 0; b; b = b->next)
		l += b->p - b->first;
	return (l);
}

static long FAR
buffers_size (int which)
{
	int	i;
	long	l;

	l = buffer_size (st.bufs[which]);
	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDD_ON))
			continue;
		l += buffer_size (st.hdd[i].bufs[which]);
	}
	return (l);
}

static int FAR
world_dynamics (void)
/*
 * Do the global housekeeping. This routine is called repeatedly from the
 * main program just before the objects simulation.
 *
 */
{
	if (st.quiet == 2)
		TnEngine[0] = iabs (CC->speed/VONE);	/* engine noise */
	else
		TnEngine[0] = 0;

	return (0);
}

extern void FAR
double_buffer (int mode)
{
	Ushort	flags;

	if (!((mode ^ st.flags1) & SF_DBUFFERING))
		return;
	if (CS->device->npages > 1) {
		flags = st.flags;
		st.flags |= SF_SIMULATING;
		Gr->CloseTextWindow (CT);
		reset_page (1);

		st.flags1 ^= SF_DBUFFERING;
		if (st.flags1 & SF_DBUFFERING) {
			Gr->SetVisual (1-st.which_buffer);
			Gr->SetVisual (st.which_buffer);
		} else {
			Gr->SetVisual (0);
			Gr->SetActive (0);
		}

		Gr->OpenTextWindow (CT);
		clear_text ();				/* clear */
		Gr->SetTextPos (2, 1);
		show_fixed (0);				/* show */

		st.flags = flags;
		MsgPrintf (50, "%s buffering",
			st.flags1 & SF_DBUFFERING ? "double" : "single");
	} else
		MsgPrintf (50, "no double buffering");
}

extern void FAR
reset_page (int empty)
{
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);
		if (Gr->Clear)
			Gr->Clear (CS);
		else {
			erase_buffers (0);
			erase_buffers (1);
		}
		Gr->SetActive (1-st.which_buffer);
	}
	if (Gr->Clear)
		Gr->Clear (CS);
	else {
		erase_buffers (0);
		erase_buffers (1);
	}
	if (empty) {
		free_buffers (0);
		free_buffers (1);
	}
	if (!Gr->Clear)
		show_fixed (1);
}

static void FAR
draw_border (int orgx, int orgy, int sizex, int sizey, int c)
{
	Gr->MoveTo (orgx - sizex, orgy - sizey);
	Gr->DrawTo (orgx + sizex, orgy - sizey, c);
	Gr->DrawTo (orgx + sizex, orgy + sizey, c);
	Gr->DrawTo (orgx - sizex, orgy + sizey, c);
	Gr->DrawTo (orgx - sizex, orgy - sizey, c);
}

static void FAR
show_border (VIEW *v, int c, int dx)
{
	int	orgx, orgy, sizex, sizey;

	orgx  = fmul (v->window->orgx, v->screen->sizex) + v->screen->minx;
	orgy  = fmul (v->window->orgy, v->screen->sizey) + v->screen->miny;
	sizex = fmul (v->window->maxx, v->screen->sizex);
	sizey = fmul (v->window->maxy, v->screen->sizey);

	draw_border (orgx, orgy, sizex, sizey, c);
	if (dx)
		draw_border (orgx+dx, orgy, sizex, sizey, c);
}

static void FAR
show_borders (int c)
{
	int	dx, i;

	dx = (st.stereo == 1)
		? fmul (muldiv(CW->maxx, 2*st.gap+2, st.gap-1), CS->sizex)
		: 0;

	show_border (CVIEW, c, dx);

	for (i = 0; i < NHDD; ++i) {
		if (st.hdd[i].view.window)
			show_border (&st.hdd[i].view, c, 0);
	}
}

static void FAR
show_fixed_page (int del)
{
	if (st.flags & SF_BLANKER)
		return;

	show_borders (del ? CS->BgColor : CS->BoColor);
}

extern void FAR
show_fixed (int del)
{
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);
		show_fixed_page (del);
		Gr->SetActive (1-st.which_buffer);
	}
	show_fixed_page (del);
}

static void FAR
render_bg (OBJECT *pov, int frame)
{
	int	t, skycolor;

	if (frame) {				/* show bg at infinity */
		t = muldiv (CP->z, CP->shift, CP->distz);
		t = muldiv (CW->maxx, t, CP->maxx) / 2;
		skycolor = (frame==1) ? st.red : st.blue;
	} else {
		t = 0;
		skycolor = st.lblue;
	}

	CW->orgx += t;

	if (st.flags & SF_SKY)
		show_sky (CVIEW, pov, skycolor, t);

	CW->orgx -= t;
}

static void FAR
render_fg (OBJECT *pov, int type)
{
	show_main (CVIEW, pov, 0, type);
	show_inst (CVIEW, CV);
}

extern BUFFER * FAR
new_buffer (void)
{
	BUFFER	*b;

	if (st.nbuffers >= st.maxbuffers)
		return (0);
	b = (BUFFER *) memory_alloc (sizeof (*b) +
					(BUFLEN - 1) * sizeof (BUFLINE));
	if (!b)
		return (0);

	++st.nbuffers;
	b->next = 0;
	b->size = BUFLEN;
	b->end = b->first + b->size;
	b->p = b->first;
	if (st.buf[TAIL])
		st.buf[TAIL]->next = b;
	else
		st.buf[HEAD] = b;
	st.buf[TAIL] = b;

	return (b);
}

#ifndef add_line
extern int FAR FASTCALL
add_line (int x, int y, int t)
{
	register BUFLINE	*p;
	BUFFER			*b;

	if (((b = st.buf[TAIL]) && (p = b->p) != b->end) ||
			((b = new_buffer ()) && (p = b->p))) {
		p->x = x;
		p->y = y;
		p->t = t;
		b->p++;
		return (0);
	}
	return (1);
}
#endif

extern int FAR
add_5op (int t, int a, int b, int c, int d, int e)
{
	BUFFER	*buf;

	if ((buf = st.buf[TAIL]) && (buf->p == buf->end-1))
		if (add_line (0, 0, T_NOP))		/* force new buf */
			return (1);

	add_line (a, b, t);
	add_line (c, d, e);
	return (0);
}

static void FAR
show_picture (void)
{
	int	bufold, bufnew;
	long	t;

	if (st.flags & SF_CLEARED) {
		st.flags &= ~SF_CLEARED;
		show_fixed (0);
	}

	if ((st.flags & SF_PAUSED) && (3 != st.stereo))
		goto ret;

	bufnew = st.which_buffer;
	st.which_buffer = (st.which_buffer+1)%NBUFS;
	bufold = st.which_buffer;

	t = Tm->Milli ();
	if (st.flags1 & SF_DBUFFERING) {
		Gr->SetActive (st.which_buffer);	/* access new page */
		show_buffers (bufnew);			/* draw new page */
		if (Gr->Clear)
			show_fixed (0);
		st.stats[29] = Tm->Milli ();
		Gr->SetVisual (st.which_buffer);	/* show new page */
		if (3 == st.stereo)
			Gr->Shutters (st.which_buffer);
		st.stats[29] = Tm->Milli () - st.stats[29];
		Gr->SetActive (1-st.which_buffer);	/* access old page */
		erase_buffers (bufold);			/* erase old page */
		free_buffers (bufold);			/* erase old page */
	} else {
		erase_buffers (bufold);			/* erase old screen */
		free_buffers (bufold);			/* erase old page */
		if (3 == st.stereo)
			Gr->Shutters (st.which_buffer);
		show_buffers (bufnew);			/* show new screen */
		if (Gr->Clear)
			show_fixed (0);
	}
	st.video_time = (int)(Tm->Milli () - t - st.stats[29]);
ret:
	Gr->Flush ();
}

extern OBJECT * FAR
get_viewer (int type)
{
	OBJECT	*pov;

	for (pov = CO; pov; pov = pov->next) {
		if (pov->name == O_VIEWER && pov->misc[0] == type)
			break;
	}
	return (pov);
}

extern void FAR
save_viewport (OBJECT *p)
{
	if (!p->viewport) {
		if (!NEW (p->viewport))
			return;
	}
	memcpy (p->viewport, CP, sizeof (*p->viewport));
}

extern void FAR
get_viewport (OBJECT *p)
{
	if (p->viewport)
		memcpy (CP, p->viewport, sizeof (*CP));
	else {
		CP->x    = 0*FONE;		/* viewer at center */
		CP->y    = 0*FONE;
		CP->shift = 0;
		CP->eyex = 0;
		CP->eyey = 0;
		CP->eyez = 0;
		CP->rotx = 0;
		CP->roty = 0;
		CP->rotz = 0;
	}
}

/*
 * Render the whole screen. Some data may already be in the display list
 * from spurious debug studd.
*/

static void FAR
render_picture (void)
{
	ANGLE	cross;		/* stereo cross-eyed angle */
	int	rev, frame, shift, rotz, type, hud, scene;
	OBJECT	*pov;
	long	t;

	st.flags |= SF_MAIN;			/* main display */

	if ((st.flags1 & SF_EXTVIEW)) {			/* find viewer */
		type = st.extview;
		hud = (HDD_HUD == type);
		scene = scenery (type);
		if (scene || hud) {
			pov = get_viewer (hud ? HDD_FRONT : type);
			if (!pov) {
				pov = CV;
				type = 0;
			}
		} else
			pov = CV;
	} else {
		pov = CV;
		type = 0;
		scene = 1;
	}

	cross = 0;	/*ASIN (muldiv (st.paralax, FONE, st.focus));*/

	if (st.stereo) {
		rev = st.flags1 & SF_STEREOREV;
		if (st.stereo == 3 && st.which_buffer)
			rev = !rev;
		if (rev) {
			shift = -st.paralax;	/* transverse: right eye */
			rotz  = -cross;
		} else {
			shift = st.paralax;	/* parallel:    left eye */
			rotz  = cross;
		}
		CP->shift -= shift;		/* left eye */
		CP->rotz  += rotz;
	} else
		shift = rotz = 0;		/* avoid compiler warning */

	if (st.stereo == 2) {
		st.cfg  = st.white;		/* should program palette */
		st.hfg  = st.white;
		st.hfgi = st.white;
		add_line (0, 0, T_MOR);
	} else {
		st.cfg = st.red;
		st.hfg = st.hudlow;
		st.hfgi = st.hudhigh;
		add_line (0, 0, T_MSET);
	}

	frame = (st.stereo == 2 ? 1 : 0);	/* mono/left frame */

	if (scene) {
		t = Tm->Milli ();
		render_bg (pov, frame);
		st.stats[40] += Tm->Milli () - t;
		objects_show (CVIEW, pov, frame, 0, 0);
	}

#ifdef	HUDINFRONT
	if (st.stereo) {
		CP->shift += shift;
		CP->rotz  -= rotz;
	}
#endif

	if (st.stereo != 2) {
		t = Tm->Milli ();
		render_fg (pov, type);
		st.stats[40] += Tm->Milli () - t;
	}

#ifndef	HUDINFRONT
	if (st.stereo) {
		CP->shift += shift;
		CP->rotz  -= rotz;
	}
#endif
		
	if (st.stereo && 3 != st.stereo) {
		if (st.stereo == 1)
			CW->orgx += muldiv (CW->maxx, 2*st.gap+2, st.gap-1);

		CP->shift += shift;		/* right eye */
		CP->rotz  -= rotz;

		frame = (st.stereo == 2 ? 2 : 0);	/* right frame */
		if (scene) {
			t = Tm->Milli ();
			render_bg (pov, frame);
			st.stats[40] += Tm->Milli () - t;
			objects_show (CVIEW, pov, frame, 0, 0);
		}
		add_line (0, 0, T_MSET);

#ifdef	HUDINFRONT
		CP->shift -= shift;
		CP->rotz  += rotz;
#endif
		t = Tm->Milli ();
		render_fg (pov, type);
		st.stats[40] += Tm->Milli () - t;

#ifndef	HUDINFRONT
		CP->shift -= shift;
		CP->rotz  += rotz;
#endif
		if (st.stereo == 1)
			CW->orgx -= muldiv(CW->maxx,2*st.gap+2,st.gap-1);
	}

	st.flags &= ~SF_MAIN;			/* secondary displays */

	st.bufs[st.which_buffer] = st.buf[HEAD];
	st.buf[HEAD] = st.buf[TAIL] = 0;

	show_hdd ();

	st.stats[26] = buffers_size (st.which_buffer);
}
